/*:
 * @target MZ
 * @plugindesc 【軽量＆厳密】ピクチャ移動（十字キー／px指定）＋「画面内クランプ」安定版。Live2D等の非ビットマップ用に実寸指定フォールバックを追加。
 * @author you
 * @url 
 *
 * @help
 * ▼概要
 *  十字キーでピクチャを滑らかに移動させるテスト用プラグイン。
 *  クランプ（画面外へ出さない）計算を強化し、
 *  画像ビットマップが取得できない（例：Live2D）場合でも、
 *  手動で実寸（幅/高さ）を登録すれば正しく画面内に収まります。
 *
 * ▼新規コマンド
 *  @command setContentSize
 *  @text コンテンツ実寸(幅/高)の手動設定（Live2Dなど用）
 *  @desc ビットマップが無いピクチャのクランプ計算に用いる元サイズ。
 *  @arg id
 *  @text ピクチャID
 *  @type number
 *  @min 1
 *  @default 1
 *  @arg width
 *  @text 幅(px)
 *  @type number
 *  @min 1
 *  @default 1920
 *  @arg height
 *  @text 高(px)
 *  @type number
 *  @min 1
 *  @default 1080
 *
 * ▼既存の主なコマンド（例）
 *   movePixels: 指定ピクセルぶん移動/時間指定
 *   enable/disable: キー操作の有効/無効
 *   quickSetupAll: よく使う設定を一括で適用
 *
 * ▼使い方の例
 *   1) Live2DをピクチャID=1で表示
 *   2) 一度だけ下記を実行（見かけの基準サイズを登録）
 *      setContentSize id=1 width=1920 height=1080
 *   3) 以後は縮小拡大しても正しくクランプが効きます
 *
 * このプラグインはテスト目的の簡易実装です。商用・非商用問わずご利用可。
 */

(()=>{
  "use strict";
  const PN = "PictureKeyMoveTest";

  // 画面サイズの取得関数
  const SCR_W = ()=> Graphics.width;
  const SCR_H = ()=> Graphics.height;

  // 状態
  const state = {
    enabled: true,
    allowDuringMessage: false,
    step: 16,         // 1回の移動量(px)
    duration: 6,      // フレーム数
    diag: false,      // 斜め移動許可
    clamp: true,      // 画面内クランプ
    targetIds: [1],   // 対象のピクチャID配列
  };

  // --- Added: manual content-size registry for non-bitmap pictures (e.g., Live2D) ---
  const _pkmtContentSizeById = {}; // { [pictureId:number]: {w:number,h:number} }
  function _pkmtGetOverrideSizeFor(pic){
    try{
      const id = pic?._pictureId ?? pic?._id;
      if(id && _pkmtContentSizeById[id]) return _pkmtContentSizeById[id];
    }catch(_){ }
    return null;
  }

  //==============================
  // ユーティリティ
  //==============================
  function isMsgBusy(){
    if(!$gameMessage) return false;
    return $gameMessage.isBusy();
  }

  function pictureOf(id){
    return $gameScreen.picture(id);
  }

  function currentKeyVector(){
    // 十字キーからベクトルを作る
    const l = Input.isPressed("left");
    const r = Input.isPressed("right");
    const u = Input.isPressed("up");
    const d = Input.isPressed("down");
    let x = (l? -1:0) + (r? 1:0);
    let y = (u? -1:0) + (d? 1:0);
    if(!state.diag){
      // 斜め禁止: 優先度 Left/Right > Up/Down
      if(x!==0) y = 0;
    }
    return {x,y};
  }

  //==============================
  // クランプ境界の計算
  //==============================
  function computeBoundsForPicture(pic){
    const name = pic._name||"";
    const id   = pic?._pictureId ?? pic?._id;
    // Base size candidates:
    let baseW = 0, baseH = 0;

    // 1) Try normal bitmap (pictures/ folder)
    if(name){
      const bmp = ImageManager.loadPicture(name);
      if(bmp && bmp.isReady()){
        baseW = bmp.width||0;
        baseH = bmp.height||0;
      }
    }

    // 2) Fallback: user-provided override (for Live2D etc.)
    if((!baseW || !baseH) && id){
      const ov = _pkmtGetOverrideSizeFor(pic);
      if(ov){ baseW = ov.w|0; baseH = ov.h|0; }
    }

    if(!baseW || !baseH){
      // No size yet (e.g., bitmap not loaded & no override) → retry later
      pic._pkmtBounds = null;
      return;
    }

    const scrW = SCR_W(), scrH = SCR_H();
    const origin = pic._origin||0;
    const sx = ((pic._scaleX!=null?pic._scaleX:100)/100)||1;
    const sy = ((pic._scaleY!=null?pic._scaleY:100)/100)||1;
    const w = baseW * sx, h = baseH * sy;

    let minX, maxX, minY, maxY;
    if(origin===0){
      // 左上原点：小さい軸は固定(0)。大きい軸は [scr - img, 0]
      minX = (w<=scrW) ? 0 : Math.floor(scrW - w);
      maxX = 0;
      minY = (h<=scrH) ? 0 : Math.floor(scrH - h);
      maxY = 0;
    }else{
      // 中央原点：小さい軸は中央固定。大きい軸は [半分, 画面-半分]
      const hw=w/2, hh=h/2;
      minX = (w<=scrW) ? Math.round(scrW/2) : Math.ceil(hw);
      maxX = (w<=scrW) ? Math.round(scrW/2) : Math.floor(scrW - hw);
      minY = (h<=scrH) ? Math.round(scrH/2) : Math.ceil(hh);
      maxY = (h<=scrH) ? Math.round(scrH/2) : Math.floor(scrH - hh);
    }
    pic._pkmtBounds = {minX,maxX,minY,maxY};
  }

  function ensureBounds(pic){
    if(!pic) return;
    if(!pic._pkmtBounds) computeBoundsForPicture(pic);
  }

  function clampXY(pic, x, y){
    ensureBounds(pic);
    const b = pic._pkmtBounds;
    if(!b) return {x,y};
    x = Math.min(b.maxX, Math.max(b.minX, x));
    y = Math.min(b.maxY, Math.max(b.minY, y));
    return {x,y};
  }

  //==============================
  // 入力→移動
  //==============================
  function movePicBy(id, dx, dy, dur){
    const pic = pictureOf(id);
    if(!pic) return;
    const x = (pic._x||0)+dx;
    const y = (pic._y||0)+dy;
    const clamped = state.clamp ? clampXY(pic, x, y) : {x,y};
    $gameScreen.movePicture(id, pic._origin||0, clamped.x, clamped.y, pic._scaleX||100, pic._scaleY||100, pic._opacity||255, pic._blendMode||0, dur);
  }

  //==============================
  // Game_Screen 拡張（毎フレームの押しっぱ処理用）
  //==============================
  const _Game_Screen_update = Game_Screen.prototype.update;
  Game_Screen.prototype.update = function(){
    _Game_Screen_update.call(this);
    if(!state.enabled) return;
    if(!state.allowDuringMessage && isMsgBusy()) return;

    // 入力ベクトル
    const v = currentKeyVector();
    if(v.x===0 && v.y===0) return;

    // 対象群に一括適用
    const step = state.step;
    const dur  = state.duration;
    const dx = v.x * step;
    const dy = v.y * step;
    for(const id of state.targetIds){
      movePicBy(id, dx, dy, dur);
    }
  };

  //==============================
  // プラグインコマンド
  //==============================
  PluginManager.registerCommand(PN, "enable", args=>{
    state.enabled = true;
  });
  PluginManager.registerCommand(PN, "disable", args=>{
    state.enabled = false;
  });
  PluginManager.registerCommand(PN, "allowMessage", args=>{
    state.allowDuringMessage = String(args.value||"off").toLowerCase()==="on";
  });
  PluginManager.registerCommand(PN, "setStep", args=>{
    state.step = Math.max(1, Number(args.value||16)|0);
  });
  PluginManager.registerCommand(PN, "setDuration", args=>{
    state.duration = Math.max(1, Number(args.value||6)|0);
  });
  PluginManager.registerCommand(PN, "setClamp", args=>{
    state.clamp = String(args.value||"on").toLowerCase()==="on";
  });
  PluginManager.registerCommand(PN, "setTargets", args=>{
    // CSV → number[]
    const csv = String(args.ids||"1");
    state.targetIds = csv.split(/[,\s]+/).map(s=>Number(s)|0).filter(n=>n>0);
  });

  PluginManager.registerCommand(PN, "movePixels", args=>{
    const id = Math.max(1, Number(args.id||1)|0);
    const dx = Number(args.dx||0)|0;
    const dy = Number(args.dy||0)|0;
    const dur = Math.max(1, Number(args.duration||6)|0);
    movePicBy(id, dx, dy, dur);
  });

  // 新規：Live2D等のための実寸登録
  PluginManager.registerCommand(PN, "setContentSize", args=>{
    const id = Math.max(1, Number(args.id||0)|0);
    const w  = Math.max(1, Number(args.width||0)|0);
    const h  = Math.max(1, Number(args.height||0)|0);
    _pkmtContentSizeById[id] = {w,h};
    const pic = $gameScreen.picture(id);
    if(pic){ computeBoundsForPicture(pic); }
  });

  //==============================
  // クイック設定（例）
  //==============================
  PluginManager.registerCommand(PN, "quickSetupAll", args=>{
    state.enabled = true;
    state.allowDuringMessage = String(args.allowMsg||"on").toLowerCase()==="on";
    state.step = Math.max(1, Number(args.step||16)|0);
    state.duration = Math.max(1, Number(args.dur||6)|0);
    state.clamp = String(args.clamp||"on").toLowerCase()==="on";
    state.diag = String(args.diag||"off").toLowerCase()==="on";
    const csv = String(args.ids||"1");
    state.targetIds = csv.split(/[,\s]+/).map(s=>Number(s)|0).filter(n=>n>0);
  });

})();
